import { z } from "zod";
import { insertUserSchema, insertCustomerSchema, insertServiceRequestSchema, createInvoiceWithItemsSchema, users, customers, serviceRequests, invoices, invoiceItems } from "./schema";

export * from "./schema";

export const errorSchemas = {
  validation: z.object({
    message: z.string(),
    field: z.string().optional(),
  }),
  notFound: z.object({
    message: z.string(),
  }),
  unauthorized: z.object({
    message: z.string(),
  }),
};

// We use z.any() here to handle the nested items structure without writing a complex custom zod schema mapping again,
// but the frontend will typecast it nicely.
const invoiceWithItemsSchema = z.any(); 

export const api = {
  auth: {
    login: {
      method: "POST" as const,
      path: "/api/login" as const,
      input: insertUserSchema,
      responses: {
        200: z.custom<typeof users.$inferSelect>(),
        401: errorSchemas.unauthorized,
      }
    },
    logout: {
      method: "POST" as const,
      path: "/api/logout" as const,
      responses: {
        200: z.object({ message: z.string() })
      }
    },
    me: {
      method: "GET" as const,
      path: "/api/me" as const,
      responses: {
        200: z.custom<typeof users.$inferSelect>(),
        401: errorSchemas.unauthorized,
      }
    }
  },
  serviceRequests: {
    list: {
      method: "GET" as const,
      path: "/api/service-requests" as const,
      responses: {
        200: z.array(z.custom<typeof serviceRequests.$inferSelect>()),
      }
    },
    create: {
      method: "POST" as const,
      path: "/api/service-requests" as const,
      input: insertServiceRequestSchema,
      responses: {
        201: z.custom<typeof serviceRequests.$inferSelect>(),
        400: errorSchemas.validation,
      }
    },
    update: {
      method: "PUT" as const,
      path: "/api/service-requests/:id" as const,
      input: z.object({ status: z.string() }),
      responses: {
        200: z.custom<typeof serviceRequests.$inferSelect>(),
        404: errorSchemas.notFound,
      }
    },
    delete: {
      method: "DELETE" as const,
      path: "/api/service-requests/:id" as const,
      responses: {
        204: z.void(),
        404: errorSchemas.notFound,
      }
    }
  },
  customers: {
    list: {
      method: "GET" as const,
      path: "/api/customers" as const,
      responses: {
        200: z.array(z.custom<typeof customers.$inferSelect>()),
      }
    },
    create: {
      method: "POST" as const,
      path: "/api/customers" as const,
      input: insertCustomerSchema,
      responses: {
        201: z.custom<typeof customers.$inferSelect>(),
        400: errorSchemas.validation,
      }
    },
    update: {
      method: "PUT" as const,
      path: "/api/customers/:id" as const,
      input: insertCustomerSchema.partial(),
      responses: {
        200: z.custom<typeof customers.$inferSelect>(),
        404: errorSchemas.notFound,
      }
    },
    delete: {
      method: "DELETE" as const,
      path: "/api/customers/:id" as const,
      responses: {
        204: z.void(),
        404: errorSchemas.notFound,
      }
    }
  },
  invoices: {
    list: {
      method: "GET" as const,
      path: "/api/invoices" as const,
      responses: {
        200: z.array(invoiceWithItemsSchema),
      }
    },
    create: {
      method: "POST" as const,
      path: "/api/invoices" as const,
      input: createInvoiceWithItemsSchema,
      responses: {
        201: invoiceWithItemsSchema,
        400: errorSchemas.validation,
      }
    },
    delete: {
      method: "DELETE" as const,
      path: "/api/invoices/:id" as const,
      responses: {
        204: z.void(),
        404: errorSchemas.notFound,
      }
    }
  },
  system: {
    export: {
      method: "GET" as const,
      path: "/api/system/export" as const,
      responses: {
        200: z.any(),
        401: errorSchemas.unauthorized,
      }
    }
  }
};

export function buildUrl(path: string, params?: Record<string, string | number>): string {
  let url = path;
  if (params) {
    Object.entries(params).forEach(([key, value]) => {
      if (url.includes(`:${key}`)) {
        url = url.replace(`:${key}`, String(value));
      }
    });
  }
  return url;
}