import {
  users, customers, serviceRequests, invoices, invoiceItems,
  type User, type InsertUser,
  type Customer, type InsertCustomer,
  type ServiceRequest, type InsertServiceRequest,
  type Invoice, type InsertInvoice, type InvoiceWithItems,
  type InvoiceItem
} from "@shared/schema";
import { db } from "./db";
import { eq } from "drizzle-orm";
import session from "express-session";
import connectPg from "connect-pg-simple";
import { pool } from "./db";

const PostgresSessionStore = connectPg(session);

export function setupSession(app: any) {
  app.use(
    session({
      store: new PostgresSessionStore({ pool, createTableIfMissing: true }),
      secret: process.env.SESSION_SECRET || "super_secret_for_now",
      resave: false,
      saveUninitialized: false,
      cookie: { maxAge: 30 * 24 * 60 * 60 * 1000 }, // 30 days
    })
  );
}

export interface IStorage {
  getUserByUsername(username: string): Promise<User | undefined>;
  getUser(id: number): Promise<User | undefined>;
  createUser(user: InsertUser): Promise<User>;

  getCustomers(): Promise<Customer[]>;
  getCustomer(id: number): Promise<Customer | undefined>;
  createCustomer(customer: InsertCustomer): Promise<Customer>;
  updateCustomer(id: number, updates: Partial<InsertCustomer>): Promise<Customer | undefined>;
  deleteCustomer(id: number): Promise<void>;

  getServiceRequests(): Promise<ServiceRequest[]>;
  getServiceRequest(id: number): Promise<ServiceRequest | undefined>;
  createServiceRequest(request: InsertServiceRequest): Promise<ServiceRequest>;
  updateServiceRequestStatus(id: number, status: string): Promise<ServiceRequest | undefined>;
  deleteServiceRequest(id: number): Promise<void>;

  getInvoices(): Promise<InvoiceWithItems[]>;
  createInvoiceWithItems(invoice: InsertInvoice, items: Omit<InvoiceItem, "id" | "invoiceId">[]): Promise<InvoiceWithItems>;
  deleteInvoice(id: number): Promise<void>;

  exportDatabase(): Promise<any>;
}

export class DatabaseStorage implements IStorage {
  async getUserByUsername(username: string): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.username, username));
    return user;
  }
  async getUser(id: number): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.id, id));
    return user;
  }
  async createUser(insertUser: InsertUser): Promise<User> {
    const [user] = await db.insert(users).values(insertUser).returning();
    return user;
  }

  async getCustomers(): Promise<Customer[]> {
    return await db.select().from(customers);
  }
  async getCustomer(id: number): Promise<Customer | undefined> {
    const [customer] = await db.select().from(customers).where(eq(customers.id, id));
    return customer;
  }
  async createCustomer(customer: InsertCustomer): Promise<Customer> {
    const [created] = await db.insert(customers).values(customer).returning();
    return created;
  }
  async updateCustomer(id: number, updates: Partial<InsertCustomer>): Promise<Customer | undefined> {
    const [updated] = await db.update(customers).set(updates).where(eq(customers.id, id)).returning();
    return updated;
  }
  async deleteCustomer(id: number): Promise<void> {
    await db.delete(customers).where(eq(customers.id, id));
  }

  async getServiceRequests(): Promise<ServiceRequest[]> {
    return await db.select().from(serviceRequests);
  }
  async getServiceRequest(id: number): Promise<ServiceRequest | undefined> {
    const [req] = await db.select().from(serviceRequests).where(eq(serviceRequests.id, id));
    return req;
  }
  async createServiceRequest(request: InsertServiceRequest): Promise<ServiceRequest> {
    const [created] = await db.insert(serviceRequests).values(request).returning();
    return created;
  }
  async updateServiceRequestStatus(id: number, status: string): Promise<ServiceRequest | undefined> {
    const [updated] = await db.update(serviceRequests).set({ status }).where(eq(serviceRequests.id, id)).returning();
    return updated;
  }
  async deleteServiceRequest(id: number): Promise<void> {
    await db.delete(serviceRequests).where(eq(serviceRequests.id, id));
  }

  async getInvoices(): Promise<InvoiceWithItems[]> {
    const allInvoices = await db.select().from(invoices);
    const allItems = await db.select().from(invoiceItems);
    
    return allInvoices.map(inv => ({
      ...inv,
      items: allItems.filter(item => item.invoiceId === inv.id)
    }));
  }
  async createInvoiceWithItems(invoice: InsertInvoice, items: Omit<InvoiceItem, "id" | "invoiceId">[]): Promise<InvoiceWithItems> {
    const [createdInvoice] = await db.insert(invoices).values(invoice).returning();
    
    const itemsToInsert = items.map(item => ({
      ...item,
      invoiceId: createdInvoice.id
    }));
    
    const createdItems = await db.insert(invoiceItems).values(itemsToInsert).returning();
    
    return {
      ...createdInvoice,
      items: createdItems
    };
  }
  async deleteInvoice(id: number): Promise<void> {
    await db.delete(invoices).where(eq(invoices.id, id));
  }

  async exportDatabase(): Promise<any> {
    const allUsers = await db.select().from(users);
    const allCustomers = await db.select().from(customers);
    const allServiceRequests = await db.select().from(serviceRequests);
    const allInvoices = await db.select().from(invoices);
    const allInvoiceItems = await db.select().from(invoiceItems);

    return {
      users: allUsers,
      customers: allCustomers,
      serviceRequests: allServiceRequests,
      invoices: allInvoices,
      invoiceItems: allInvoiceItems,
    };
  }
}

export const storage = new DatabaseStorage();