import type { Express, Request, Response, NextFunction } from "express";
import { createServer, type Server } from "http";
import { storage, setupSession } from "./storage";
import { api } from "@shared/routes";
import { z } from "zod";

function requireAuth(req: Request, res: Response, next: NextFunction) {
  if (!(req.session as any).userId) {
    return res.status(401).json({ message: "Unauthorized" });
  }
  next();
}

export async function registerRoutes(
  httpServer: Server,
  app: Express
): Promise<Server> {
  setupSession(app);

  app.post(api.auth.login.path, async (req, res) => {
    try {
      const { username, password } = api.auth.login.input.parse(req.body);
      const user = await storage.getUserByUsername(username);
      
      // Exact match login since we aren't using passport for extreme portability
      if (!user || user.password !== password) {
        return res.status(401).json({ message: "Invalid credentials" });
      }
      
      (req.session as any).userId = user.id;
      res.json(user);
    } catch (err) {
      if (err instanceof z.ZodError) {
        res.status(400).json({
          message: err.errors[0].message,
          field: err.errors[0].path.join('.'),
        });
      } else {
        res.status(500).json({ message: "Internal server error" });
      }
    }
  });

  app.post(api.auth.logout.path, (req, res) => {
    req.session.destroy((err) => {
      if (err) return res.status(500).json({ message: "Failed to logout" });
      res.json({ message: "Logged out" });
    });
  });

  app.get(api.auth.me.path, async (req, res) => {
    const userId = (req.session as any).userId;
    if (!userId) {
      return res.status(401).json({ message: "Unauthorized" });
    }
    const user = await storage.getUser(userId);
    if (!user) {
      return res.status(401).json({ message: "Unauthorized" });
    }
    res.json(user);
  });

  app.get(api.serviceRequests.list.path, requireAuth, async (req, res) => {
    const requests = await storage.getServiceRequests();
    res.json(requests);
  });

  app.post(api.serviceRequests.create.path, async (req, res) => {
    try {
      const input = api.serviceRequests.create.input.parse(req.body);
      const request = await storage.createServiceRequest(input);
      res.status(201).json(request);
    } catch (err) {
      if (err instanceof z.ZodError) {
        res.status(400).json({
          message: err.errors[0].message,
          field: err.errors[0].path.join('.'),
        });
      }
    }
  });

  app.put(api.serviceRequests.update.path, requireAuth, async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const { status } = api.serviceRequests.update.input.parse(req.body);
      const request = await storage.updateServiceRequestStatus(id, status);
      if (!request) return res.status(404).json({ message: "Not found" });
      res.json(request);
    } catch (err) {
      if (err instanceof z.ZodError) {
        res.status(400).json({ message: "Validation error" });
      }
    }
  });

  app.delete(api.serviceRequests.delete.path, requireAuth, async (req, res) => {
    const id = parseInt(req.params.id);
    await storage.deleteServiceRequest(id);
    res.status(204).end();
  });

  app.get(api.customers.list.path, requireAuth, async (req, res) => {
    const customers = await storage.getCustomers();
    res.json(customers);
  });

  app.post(api.customers.create.path, requireAuth, async (req, res) => {
    try {
      const input = api.customers.create.input.parse(req.body);
      const customer = await storage.createCustomer(input);
      res.status(201).json(customer);
    } catch (err) {
      if (err instanceof z.ZodError) {
        res.status(400).json({ message: "Validation error" });
      }
    }
  });

  app.put(api.customers.update.path, requireAuth, async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const input = api.customers.update.input.parse(req.body);
      const customer = await storage.updateCustomer(id, input);
      if (!customer) return res.status(404).json({ message: "Not found" });
      res.json(customer);
    } catch (err) {
      if (err instanceof z.ZodError) {
        res.status(400).json({ message: "Validation error" });
      }
    }
  });

  app.delete(api.customers.delete.path, requireAuth, async (req, res) => {
    const id = parseInt(req.params.id);
    await storage.deleteCustomer(id);
    res.status(204).end();
  });

  app.get(api.invoices.list.path, requireAuth, async (req, res) => {
    const invoices = await storage.getInvoices();
    res.json(invoices);
  });

  app.post(api.invoices.create.path, requireAuth, async (req, res) => {
    try {
      const input = api.invoices.create.input.parse(req.body);
      const { items, ...invoiceData } = input;
      
      const invoice = await storage.createInvoiceWithItems(
        { ...invoiceData, taxRate: invoiceData.taxRate.toString(), totalAmount: invoiceData.totalAmount.toString() }, 
        items.map(i => ({ ...i, price: i.price.toString() }))
      );
      res.status(201).json(invoice);
    } catch (err) {
      if (err instanceof z.ZodError) {
        res.status(400).json({ message: "Validation error" });
      }
    }
  });

  app.delete(api.invoices.delete.path, requireAuth, async (req, res) => {
    const id = parseInt(req.params.id);
    await storage.deleteInvoice(id);
    res.status(204).end();
  });

  app.get(api.system.export.path, requireAuth, async (req, res) => {
    const data = await storage.exportDatabase();
    res.json(data);
  });

  // Seed data
  seedDatabase();

  return httpServer;
}

async function seedDatabase() {
  const existingUser = await storage.getUserByUsername("admin");
  if (!existingUser) {
    await storage.createUser({ username: "admin", password: "admin" });
  }

  const existingCustomers = await storage.getCustomers();
  if (existingCustomers.length === 0) {
    await storage.createCustomer({
      name: "Acme Corp",
      email: "billing@acme.com",
      phone: "555-0100",
      company: "Acme Corporation"
    });
    await storage.createCustomer({
      name: "Stark Industries",
      email: "info@stark.com",
      phone: "555-0200",
      company: "Stark Industries"
    });
  }

  const existingRequests = await storage.getServiceRequests();
  if (existingRequests.length === 0) {
    await storage.createServiceRequest({
      name: "John Doe",
      email: "john@example.com",
      phone: "555-0199",
      serviceInterest: "CCTV & Surveillance",
      message: "Looking for a 10-camera setup for our warehouse."
    });
    await storage.createServiceRequest({
      name: "Jane Smith",
      email: "jane@example.com",
      phone: "555-0299",
      serviceInterest: "Access Control",
      message: "Need keycard entry for 5 doors."
    });
  }
}