import { AdminLayout } from "@/components/AdminLayout";
import { useExportDatabase } from "@/hooks/use-system";
import { Button } from "@/components/ui/button";
import { Database, Download, ShieldAlert } from "lucide-react";
import { useToast } from "@/hooks/use-toast";

export default function Settings() {
  const exportDb = useExportDatabase();
  const { toast } = useToast();

  const handleExport = () => {
    exportDb.mutate(undefined, {
      onSuccess: () => {
        toast({ title: "Export Successful", description: "Database snapshot downloaded securely." });
      },
      onError: (error) => {
        toast({ title: "Export Failed", description: error.message, variant: "destructive" });
      }
    });
  };

  return (
    <AdminLayout>
      <div className="space-y-6 max-w-3xl">
        <div>
          <h1 className="text-3xl font-display font-bold text-white uppercase tracking-wider">System Settings</h1>
          <p className="text-muted-foreground">Core configuration and data management.</p>
        </div>

        <div className="glass-panel p-8 rounded-xl border border-white/10">
          <div className="flex items-start gap-4 mb-6">
            <div className="p-3 bg-secondary/20 rounded-lg text-secondary border border-secondary/30">
              <Database className="w-6 h-6" />
            </div>
            <div>
              <h2 className="text-xl font-bold text-white">Database Operations</h2>
              <p className="text-muted-foreground mt-1 text-sm">Manage underlying SQL records and perform emergency backups.</p>
            </div>
          </div>

          <div className="p-6 bg-black/40 rounded-lg border border-white/5">
            <div className="flex items-center justify-between">
              <div>
                <h3 className="font-bold text-white flex items-center gap-2">
                  Generate Snapshot <span className="px-2 py-0.5 rounded text-[10px] bg-primary/20 text-primary border border-primary/30 uppercase tracking-widest font-mono">Restricted</span>
                </h3>
                <p className="text-sm text-muted-foreground mt-1">Download entire database as a JSON structure for cold storage.</p>
              </div>
              <Button 
                onClick={handleExport}
                disabled={exportDb.isPending}
                className="bg-white text-black hover:bg-gray-200 gap-2 font-bold"
              >
                <Download className="w-4 h-4" />
                {exportDb.isPending ? "COMPILING..." : "EXPORT DATA"}
              </Button>
            </div>
          </div>

          <div className="mt-8 p-6 bg-destructive/10 rounded-lg border border-destructive/30 flex items-start gap-4">
            <ShieldAlert className="w-6 h-6 text-destructive shrink-0" />
            <div>
              <h3 className="font-bold text-destructive">Data Integrity Warning</h3>
              <p className="text-sm text-destructive/80 mt-1">
                Exported snapshots contain sensitive client information and unencrypted financial records. 
                Ensure downloaded files are stored in a secure, air-gapped environment according to protocol Aegis-7.
              </p>
            </div>
          </div>
        </div>
      </div>
    </AdminLayout>
  );
}
