import { AdminLayout } from "@/components/AdminLayout";
import { useServiceRequests, useUpdateServiceRequestStatus, useDeleteServiceRequest } from "@/hooks/use-service-requests";
import { Button } from "@/components/ui/button";
import { Trash2, CheckCircle, Clock } from "lucide-react";
import { format } from "date-fns";

export default function Requests() {
  const { data: requests, isLoading } = useServiceRequests();
  const updateStatus = useUpdateServiceRequestStatus();
  const deleteReq = useDeleteServiceRequest();

  return (
    <AdminLayout>
      <div className="space-y-6">
        <div className="flex justify-between items-end">
          <div>
            <h1 className="text-3xl font-display font-bold text-white uppercase tracking-wider">Service Requests</h1>
            <p className="text-muted-foreground">Manage incoming client transmissions.</p>
          </div>
        </div>

        <div className="glass-panel border-white/10 rounded-xl overflow-hidden">
          <div className="overflow-x-auto">
            <table className="w-full text-sm text-left">
              <thead className="text-xs text-muted-foreground uppercase bg-black/40 border-b border-white/10 font-display">
                <tr>
                  <th className="px-6 py-4 font-medium tracking-wider">Client Data</th>
                  <th className="px-6 py-4 font-medium tracking-wider">Vector</th>
                  <th className="px-6 py-4 font-medium tracking-wider">Payload</th>
                  <th className="px-6 py-4 font-medium tracking-wider">Status</th>
                  <th className="px-6 py-4 font-medium tracking-wider text-right">Actions</th>
                </tr>
              </thead>
              <tbody className="divide-y divide-white/5">
                {isLoading ? (
                  <tr><td colSpan={5} className="px-6 py-8 text-center text-muted-foreground">Scanning frequencies...</td></tr>
                ) : requests?.length === 0 ? (
                  <tr><td colSpan={5} className="px-6 py-8 text-center text-muted-foreground">No signals detected.</td></tr>
                ) : (
                  requests?.map((req) => (
                    <tr key={req.id} className="hover:bg-white/5 transition-colors group">
                      <td className="px-6 py-4">
                        <div className="font-medium text-white">{req.name}</div>
                        <div className="text-muted-foreground">{req.email}</div>
                        <div className="text-muted-foreground text-xs">{req.phone}</div>
                        <div className="text-xs text-white/30 font-mono mt-1">{format(new Date(req.createdAt), 'yyyy-MM-dd HH:mm')}</div>
                      </td>
                      <td className="px-6 py-4">
                        <span className="inline-flex px-2 py-1 rounded bg-white/5 border border-white/10 text-xs font-mono uppercase">
                          {req.serviceInterest}
                        </span>
                      </td>
                      <td className="px-6 py-4 max-w-xs truncate" title={req.message}>
                        {req.message}
                      </td>
                      <td className="px-6 py-4">
                        <span className={`px-2 py-1 rounded text-xs font-mono uppercase ${
                          req.status === 'new' ? 'bg-primary/20 text-primary border border-primary/30' :
                          req.status === 'completed' ? 'bg-emerald-500/20 text-emerald-400 border border-emerald-500/30' :
                          'bg-secondary/20 text-secondary border border-secondary/30'
                        }`}>
                          {req.status}
                        </span>
                      </td>
                      <td className="px-6 py-4 text-right space-x-2 opacity-100 md:opacity-0 md:group-hover:opacity-100 transition-opacity">
                        {req.status !== 'completed' && (
                          <Button 
                            variant="outline" 
                            size="sm"
                            className="bg-emerald-500/10 text-emerald-500 border-emerald-500/30 hover:bg-emerald-500/20"
                            onClick={() => updateStatus.mutate({ id: req.id, status: 'completed' })}
                          >
                            <CheckCircle className="w-4 h-4 mr-1" /> Resolve
                          </Button>
                        )}
                        {req.status === 'new' && (
                          <Button 
                            variant="outline" 
                            size="sm"
                            className="bg-secondary/10 text-secondary border-secondary/30 hover:bg-secondary/20"
                            onClick={() => updateStatus.mutate({ id: req.id, status: 'in_progress' })}
                          >
                            <Clock className="w-4 h-4 mr-1" /> Process
                          </Button>
                        )}
                        <Button 
                          variant="ghost" 
                          size="icon"
                          className="text-destructive hover:bg-destructive/20 hover:text-destructive"
                          onClick={() => {
                            if(confirm("Purge record?")) deleteReq.mutate(req.id);
                          }}
                        >
                          <Trash2 className="w-4 h-4" />
                        </Button>
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </div>
      </div>
    </AdminLayout>
  );
}
