import { useState } from "react";
import { AdminLayout } from "@/components/AdminLayout";
import { useInvoices, useDeleteInvoice } from "@/hooks/use-invoices";
import { Button } from "@/components/ui/button";
import { Trash2, FilePlus, Printer, Eye } from "lucide-react";
import { format } from "date-fns";
import { Link } from "wouter";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";

export default function Invoices() {
  const { data: invoices, isLoading } = useInvoices();
  const deleteInvoice = useDeleteInvoice();
  
  // State for print preview dialog
  const [previewInvoice, setPreviewInvoice] = useState<any | null>(null);

  const handlePrint = () => {
    window.print();
  };

  return (
    <AdminLayout>
      <div className="space-y-6 no-print">
        <div className="flex justify-between items-end">
          <div>
            <h1 className="text-3xl font-display font-bold text-white uppercase tracking-wider">Financial Ledger</h1>
            <p className="text-muted-foreground">Manage service contracts and billing.</p>
          </div>
          <Link href="/admin/invoices/new" className="inline-flex items-center justify-center px-4 py-2 rounded-md bg-secondary hover:bg-secondary/90 text-white font-bold gap-2 text-sm transition-colors">
            <FilePlus className="w-4 h-4" /> GENERATE INVOICE
          </Link>
        </div>

        <div className="glass-panel border-white/10 rounded-xl overflow-hidden">
          <div className="overflow-x-auto">
            <table className="w-full text-sm text-left">
              <thead className="text-xs text-muted-foreground uppercase bg-black/40 border-b border-white/10 font-display">
                <tr>
                  <th className="px-6 py-4 font-medium tracking-wider">ID</th>
                  <th className="px-6 py-4 font-medium tracking-wider">Entity</th>
                  <th className="px-6 py-4 font-medium tracking-wider">Date</th>
                  <th className="px-6 py-4 font-medium tracking-wider text-right">Total Amount</th>
                  <th className="px-6 py-4 font-medium tracking-wider text-center">Status</th>
                  <th className="px-6 py-4 font-medium tracking-wider text-right">Actions</th>
                </tr>
              </thead>
              <tbody className="divide-y divide-white/5">
                {isLoading ? (
                  <tr><td colSpan={6} className="px-6 py-8 text-center text-muted-foreground">Accessing ledger...</td></tr>
                ) : invoices?.length === 0 ? (
                  <tr><td colSpan={6} className="px-6 py-8 text-center text-muted-foreground">No records found.</td></tr>
                ) : (
                  invoices?.map((inv) => (
                    <tr key={inv.id} className="hover:bg-white/5 transition-colors group">
                      <td className="px-6 py-4 font-mono text-secondary">
                        INV-{inv.id.toString().padStart(4, '0')}
                      </td>
                      <td className="px-6 py-4 font-bold text-white">
                        {inv.customerName}
                      </td>
                      <td className="px-6 py-4 text-muted-foreground">
                        {format(new Date(inv.date), 'MMM dd, yyyy')}
                      </td>
                      <td className="px-6 py-4 text-right font-mono text-white">
                        ${Number(inv.totalAmount).toLocaleString(undefined, {minimumFractionDigits: 2})}
                      </td>
                      <td className="px-6 py-4 text-center">
                         <span className="px-2 py-1 rounded bg-white/10 border border-white/20 text-xs font-mono uppercase text-white/70">
                          {inv.status}
                        </span>
                      </td>
                      <td className="px-6 py-4 text-right space-x-2">
                        <Button 
                          variant="outline" 
                          size="sm"
                          className="bg-primary/10 text-primary border-primary/30 hover:bg-primary/20"
                          onClick={() => setPreviewInvoice(inv)}
                        >
                          <Eye className="w-4 h-4 mr-1" /> View/Print
                        </Button>
                        <Button 
                          variant="ghost" 
                          size="icon"
                          className="text-destructive hover:bg-destructive/20 hover:text-destructive"
                          onClick={() => {
                            if(confirm("Purge financial record?")) deleteInvoice.mutate(inv.id);
                          }}
                        >
                          <Trash2 className="w-4 h-4" />
                        </Button>
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </div>
      </div>

      {/* View/Print Dialog (Visible on screen) */}
      <Dialog open={!!previewInvoice} onOpenChange={(open) => !open && setPreviewInvoice(null)}>
        <DialogContent className="max-w-3xl bg-card border-white/10 text-foreground max-h-[90vh] overflow-y-auto no-print">
          <DialogHeader className="flex flex-row justify-between items-center border-b border-white/10 pb-4">
            <DialogTitle className="font-display text-xl uppercase tracking-wider">Document Preview</DialogTitle>
            <Button onClick={handlePrint} className="bg-white text-black hover:bg-gray-200">
              <Printer className="w-4 h-4 mr-2" /> Print Document
            </Button>
          </DialogHeader>
          
          <div className="p-8 bg-white text-black rounded-lg mt-4 border border-gray-200 shadow-sm relative" id="print-content-preview">
            <div className="flex justify-between items-start mb-12">
              <div>
                <h1 className="font-display font-black text-3xl tracking-widest uppercase">AegisNet</h1>
                <p className="text-gray-500 text-sm mt-1">Security Systems Inc.</p>
              </div>
              <div className="text-right">
                <h2 className="text-4xl font-light text-gray-300 uppercase tracking-widest">INVOICE</h2>
                <p className="font-mono mt-2 font-bold text-lg">INV-{previewInvoice?.id.toString().padStart(4, '0')}</p>
              </div>
            </div>

            <div className="grid grid-cols-2 gap-8 mb-12">
              <div>
                <p className="text-xs text-gray-400 font-bold uppercase tracking-wider mb-2">Billed To</p>
                <p className="font-bold text-xl">{previewInvoice?.customerName}</p>
              </div>
              <div className="text-right">
                <p className="text-xs text-gray-400 font-bold uppercase tracking-wider mb-2">Date of Issue</p>
                <p className="font-medium">{previewInvoice ? format(new Date(previewInvoice.date), 'MMMM dd, yyyy') : ''}</p>
              </div>
            </div>

            <table className="w-full mb-12">
              <thead>
                <tr className="border-b-2 border-black text-left uppercase text-xs tracking-wider">
                  <th className="py-3 font-bold">Description</th>
                  <th className="py-3 text-center font-bold">Qty</th>
                  <th className="py-3 text-right font-bold">Unit Price</th>
                  <th className="py-3 text-right font-bold">Total</th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200">
                {previewInvoice?.items?.map((item: any, idx: number) => (
                  <tr key={idx}>
                    <td className="py-4">{item.description}</td>
                    <td className="py-4 text-center">{item.quantity}</td>
                    <td className="py-4 text-right font-mono">${Number(item.price).toLocaleString(undefined, {minimumFractionDigits: 2})}</td>
                    <td className="py-4 text-right font-mono font-bold">${(item.quantity * Number(item.price)).toLocaleString(undefined, {minimumFractionDigits: 2})}</td>
                  </tr>
                ))}
              </tbody>
            </table>

            <div className="flex justify-end">
              <div className="w-1/2 space-y-3">
                <div className="flex justify-between text-gray-600">
                  <span>Subtotal</span>
                  <span className="font-mono">${(Number(previewInvoice?.totalAmount) / (1 + Number(previewInvoice?.taxRate)/100)).toLocaleString(undefined, {minimumFractionDigits: 2})}</span>
                </div>
                <div className="flex justify-between text-gray-600">
                  <span>Tax ({previewInvoice?.taxRate}%)</span>
                  <span className="font-mono">${(Number(previewInvoice?.totalAmount) - (Number(previewInvoice?.totalAmount) / (1 + Number(previewInvoice?.taxRate)/100))).toLocaleString(undefined, {minimumFractionDigits: 2})}</span>
                </div>
                <div className="flex justify-between border-t-2 border-black pt-3 text-xl font-bold">
                  <span>Total Due</span>
                  <span className="font-mono">${Number(previewInvoice?.totalAmount).toLocaleString(undefined, {minimumFractionDigits: 2})}</span>
                </div>
              </div>
            </div>
            
            <div className="mt-16 pt-8 border-t border-gray-200 text-center text-xs text-gray-400">
              Payment is due within 30 days. Thank you for your business.
            </div>
          </div>
        </DialogContent>
      </Dialog>

      {/* Actual Print Structure (Hidden until printed) */}
      {previewInvoice && (
        <div id="printable-invoice" className="hidden">
           <div className="flex justify-between items-start mb-12">
              <div>
                <h1 className="font-display font-black text-3xl tracking-widest uppercase m-0 p-0">AegisNet</h1>
                <p className="text-gray-500 text-sm mt-1 m-0 p-0">Security Systems Inc.</p>
              </div>
              <div className="text-right">
                <h2 className="text-4xl font-light text-gray-300 uppercase tracking-widest m-0 p-0">INVOICE</h2>
                <p className="font-mono mt-2 font-bold text-lg m-0 p-0">INV-{previewInvoice?.id.toString().padStart(4, '0')}</p>
              </div>
            </div>

            <div className="flex justify-between mb-12">
              <div>
                <p className="text-xs text-gray-400 font-bold uppercase tracking-wider mb-2">Billed To</p>
                <p className="font-bold text-xl">{previewInvoice?.customerName}</p>
              </div>
              <div className="text-right">
                <p className="text-xs text-gray-400 font-bold uppercase tracking-wider mb-2">Date of Issue</p>
                <p className="font-medium">{previewInvoice ? format(new Date(previewInvoice.date), 'MMMM dd, yyyy') : ''}</p>
              </div>
            </div>

            <table className="w-full mb-12 border-collapse">
              <thead>
                <tr className="border-b-2 border-black text-left uppercase text-xs tracking-wider">
                  <th className="py-3 font-bold">Description</th>
                  <th className="py-3 text-center font-bold">Qty</th>
                  <th className="py-3 text-right font-bold">Unit Price</th>
                  <th className="py-3 text-right font-bold">Total</th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200 border-b border-gray-200">
                {previewInvoice?.items?.map((item: any, idx: number) => (
                  <tr key={idx}>
                    <td className="py-4">{item.description}</td>
                    <td className="py-4 text-center">{item.quantity}</td>
                    <td className="py-4 text-right font-mono">${Number(item.price).toLocaleString(undefined, {minimumFractionDigits: 2})}</td>
                    <td className="py-4 text-right font-mono font-bold">${(item.quantity * Number(item.price)).toLocaleString(undefined, {minimumFractionDigits: 2})}</td>
                  </tr>
                ))}
              </tbody>
            </table>

            <div className="flex justify-end">
              <div className="w-1/2 space-y-3">
                <div className="flex justify-between text-gray-600">
                  <span>Subtotal</span>
                  <span className="font-mono">${(Number(previewInvoice?.totalAmount) / (1 + Number(previewInvoice?.taxRate)/100)).toLocaleString(undefined, {minimumFractionDigits: 2})}</span>
                </div>
                <div className="flex justify-between text-gray-600">
                  <span>Tax ({previewInvoice?.taxRate}%)</span>
                  <span className="font-mono">${(Number(previewInvoice?.totalAmount) - (Number(previewInvoice?.totalAmount) / (1 + Number(previewInvoice?.taxRate)/100))).toLocaleString(undefined, {minimumFractionDigits: 2})}</span>
                </div>
                <div className="flex justify-between border-t-2 border-black pt-3 text-xl font-bold">
                  <span>Total Due</span>
                  <span className="font-mono">${Number(previewInvoice?.totalAmount).toLocaleString(undefined, {minimumFractionDigits: 2})}</span>
                </div>
              </div>
            </div>
            
            <div className="mt-16 pt-8 border-t border-gray-200 text-center text-xs text-gray-400">
              Payment is due within 30 days. Thank you for securing your infrastructure with AegisNet.
            </div>
        </div>
      )}
    </AdminLayout>
  );
}
