import { AdminLayout } from "@/components/AdminLayout";
import { useForm, useFieldArray } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { createInvoiceWithItemsSchema } from "@shared/routes";
import { useCreateInvoice } from "@/hooks/use-invoices";
import { useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Plus, Trash2, ArrowLeft, Save } from "lucide-react";
import { Link } from "wouter";

export default function InvoiceCreate() {
  const [, setLocation] = useLocation();
  const createInvoice = useCreateInvoice();

  const form = useForm({
    resolver: zodResolver(createInvoiceWithItemsSchema),
    defaultValues: {
      customerName: "",
      date: new Date().toISOString().split('T')[0],
      taxRate: 0,
      totalAmount: 0,
      status: "draft",
      items: [{ description: "", quantity: 1, price: 0 }]
    }
  });

  const { fields, append, remove } = useFieldArray({
    control: form.control,
    name: "items"
  });

  // Watch fields to calculate total dynamically
  const watchItems = form.watch("items");
  const watchTaxRate = form.watch("taxRate");

  const subtotal = watchItems.reduce((sum, item) => sum + (Number(item.quantity) || 0) * (Number(item.price) || 0), 0);
  const taxAmount = subtotal * ((Number(watchTaxRate) || 0) / 100);
  const calculatedTotal = subtotal + taxAmount;

  const onSubmit = (data: any) => {
    // Override totalAmount with calculated server-side expected format
    const payload = {
      ...data,
      totalAmount: calculatedTotal
    };
    
    createInvoice.mutate(payload, {
      onSuccess: () => setLocation("/admin/invoices")
    });
  };

  return (
    <AdminLayout>
      <div className="space-y-6 max-w-4xl mx-auto">
        <div className="flex items-center gap-4">
          <Link href="/admin/invoices" className="p-2 rounded-full hover:bg-white/10 transition-colors">
            <ArrowLeft className="w-5 h-5 text-muted-foreground" />
          </Link>
          <div>
            <h1 className="text-3xl font-display font-bold text-white uppercase tracking-wider">Generate Invoice</h1>
            <p className="text-muted-foreground">Construct new financial record.</p>
          </div>
        </div>

        <div className="glass-panel p-8 rounded-xl border border-white/10">
          <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-8">
            {/* Header Info */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6 pb-8 border-b border-white/10">
              <div className="space-y-2">
                <Label>Billed Entity (Customer Name)</Label>
                <Input {...form.register("customerName")} className="bg-black/50 border-white/10 font-bold" placeholder="Weyland-Yutani Corp" />
                {form.formState.errors.customerName && <p className="text-xs text-destructive">{form.formState.errors.customerName.message as string}</p>}
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label>Date of Issue</Label>
                  <Input type="date" {...form.register("date")} className="bg-black/50 border-white/10" />
                </div>
                <div className="space-y-2">
                  <Label>Tax Rate (%)</Label>
                  <Input type="number" step="0.1" {...form.register("taxRate")} className="bg-black/50 border-white/10" />
                </div>
              </div>
            </div>

            {/* Line Items */}
            <div>
              <div className="flex items-center justify-between mb-4">
                <h3 className="font-display font-bold text-white tracking-widest uppercase text-sm">Line Items</h3>
                <Button 
                  type="button" 
                  variant="outline" 
                  size="sm" 
                  onClick={() => append({ description: "", quantity: 1, price: 0 })}
                  className="border-primary/50 text-primary hover:bg-primary/20 hover:text-primary"
                >
                  <Plus className="w-4 h-4 mr-1" /> Add Row
                </Button>
              </div>

              <div className="space-y-3">
                {fields.map((field, index) => (
                  <div key={field.id} className="flex items-start gap-4 p-4 rounded-lg bg-black/30 border border-white/5 group">
                    <div className="flex-1 space-y-2">
                      <Label className="text-xs text-muted-foreground">Description</Label>
                      <Input {...form.register(`items.${index}.description`)} className="bg-black/50 border-white/10" placeholder="Hardware component / Service phase" />
                      {form.formState.errors.items?.[index]?.description && <p className="text-xs text-destructive">Required</p>}
                    </div>
                    <div className="w-24 space-y-2">
                      <Label className="text-xs text-muted-foreground">Qty</Label>
                      <Input type="number" {...form.register(`items.${index}.quantity`)} className="bg-black/50 border-white/10 text-center" />
                    </div>
                    <div className="w-32 space-y-2">
                      <Label className="text-xs text-muted-foreground">Unit Price</Label>
                      <Input type="number" step="0.01" {...form.register(`items.${index}.price`)} className="bg-black/50 border-white/10 text-right font-mono" />
                    </div>
                    <div className="w-32 space-y-2">
                      <Label className="text-xs text-muted-foreground">Total</Label>
                      <div className="h-10 flex items-center justify-end px-3 font-mono font-bold text-white bg-black/20 rounded-md border border-transparent">
                        ${((Number(watchItems[index]?.quantity) || 0) * (Number(watchItems[index]?.price) || 0)).toFixed(2)}
                      </div>
                    </div>
                    <div className="pt-8">
                      <Button 
                        type="button" 
                        variant="ghost" 
                        size="icon" 
                        onClick={() => remove(index)}
                        disabled={fields.length === 1}
                        className="text-muted-foreground hover:text-destructive hover:bg-destructive/20"
                      >
                        <Trash2 className="w-4 h-4" />
                      </Button>
                    </div>
                  </div>
                ))}
              </div>
            </div>

            {/* Totals & Submit */}
            <div className="border-t border-white/10 pt-8 flex justify-between items-end">
              <div className="text-muted-foreground text-sm max-w-sm">
                Values will be permanently logged to the financial ledger upon submission. Ensure accuracy.
              </div>
              
              <div className="w-72 space-y-3">
                <div className="flex justify-between text-muted-foreground">
                  <span>Subtotal:</span>
                  <span className="font-mono text-white">${subtotal.toFixed(2)}</span>
                </div>
                <div className="flex justify-between text-muted-foreground">
                  <span>Tax ({watchTaxRate}%):</span>
                  <span className="font-mono text-white">${taxAmount.toFixed(2)}</span>
                </div>
                <div className="flex justify-between border-t border-white/20 pt-3 text-xl font-bold">
                  <span className="text-white">TOTAL:</span>
                  <span className="font-mono text-primary">${calculatedTotal.toFixed(2)}</span>
                </div>
                
                <Button 
                  type="submit" 
                  className="w-full mt-4 bg-primary hover:bg-primary/90 text-primary-foreground font-bold tracking-widest"
                  disabled={createInvoice.isPending}
                >
                  <Save className="w-4 h-4 mr-2" />
                  {createInvoice.isPending ? "COMMITTING..." : "COMMIT RECORD"}
                </Button>
              </div>
            </div>
          </form>
        </div>
      </div>
    </AdminLayout>
  );
}
