import { AdminLayout } from "@/components/AdminLayout";
import { useServiceRequests } from "@/hooks/use-service-requests";
import { useInvoices } from "@/hooks/use-invoices";
import { ShieldCheck, Activity, DollarSign, Users } from "lucide-react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Link } from "wouter";

export default function Dashboard() {
  const { data: requests } = useServiceRequests();
  const { data: invoices } = useInvoices();

  const totalRevenue = invoices?.reduce((sum, inv) => sum + Number(inv.totalAmount), 0) || 0;
  const newRequests = requests?.filter(r => r.status === 'new').length || 0;

  return (
    <AdminLayout>
      <div className="space-y-8">
        <div>
          <h1 className="text-3xl font-display font-bold text-white uppercase tracking-wider">Tactical Overview</h1>
          <p className="text-muted-foreground">System metrics and operational status.</p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          <Card className="bg-card/50 border-white/10">
            <CardHeader className="flex flex-row items-center justify-between pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground uppercase font-display">System Status</CardTitle>
              <Activity className="h-4 w-4 text-emerald-500" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-white">OPTIMAL</div>
              <p className="text-xs text-muted-foreground mt-1">All services online</p>
            </CardContent>
          </Card>

          <Card className="bg-card/50 border-white/10">
            <CardHeader className="flex flex-row items-center justify-between pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground uppercase font-display">Active Signals</CardTitle>
              <ShieldCheck className="h-4 w-4 text-primary" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-white">{requests?.length || 0}</div>
              <p className="text-xs text-muted-foreground mt-1">{newRequests} pending attention</p>
            </CardContent>
          </Card>

          <Card className="bg-card/50 border-white/10">
            <CardHeader className="flex flex-row items-center justify-between pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground uppercase font-display">Capital Flow</CardTitle>
              <DollarSign className="h-4 w-4 text-secondary" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-white">${totalRevenue.toLocaleString(undefined, {minimumFractionDigits: 2})}</div>
              <p className="text-xs text-muted-foreground mt-1">Total volume secured</p>
            </CardContent>
          </Card>

          <Card className="bg-card/50 border-white/10">
            <CardHeader className="flex flex-row items-center justify-between pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground uppercase font-display">Data Vault</CardTitle>
              <Users className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-white">{invoices?.length || 0}</div>
              <p className="text-xs text-muted-foreground mt-1">Secured contracts logged</p>
            </CardContent>
          </Card>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          <Card className="bg-card/50 border-white/10">
            <CardHeader className="border-b border-white/5">
              <CardTitle className="font-display">Recent Transmissions (Requests)</CardTitle>
            </CardHeader>
            <CardContent className="p-0">
              <div className="divide-y divide-white/5">
                {requests?.slice(0, 5).map(req => (
                  <div key={req.id} className="p-4 flex items-center justify-between hover:bg-white/5 transition-colors">
                    <div>
                      <p className="font-medium text-white">{req.name}</p>
                      <p className="text-sm text-muted-foreground">{req.serviceInterest}</p>
                    </div>
                    <div className="flex items-center gap-4">
                      <span className={`px-2 py-1 rounded text-xs font-mono uppercase ${
                        req.status === 'new' ? 'bg-primary/20 text-primary border border-primary/30' :
                        req.status === 'completed' ? 'bg-emerald-500/20 text-emerald-400 border border-emerald-500/30' :
                        'bg-secondary/20 text-secondary border border-secondary/30'
                      }`}>
                        {req.status}
                      </span>
                    </div>
                  </div>
                ))}
                {(!requests || requests.length === 0) && (
                  <div className="p-8 text-center text-muted-foreground font-mono">No active transmissions.</div>
                )}
              </div>
              <div className="p-4 border-t border-white/5 bg-black/20">
                <Link href="/admin/requests" className="text-sm text-secondary hover:text-white transition-colors font-medium">View All Matrix Data →</Link>
              </div>
            </CardContent>
          </Card>

          <Card className="bg-card/50 border-white/10">
            <CardHeader className="border-b border-white/5">
              <CardTitle className="font-display">Recent Financial Logs (Invoices)</CardTitle>
            </CardHeader>
            <CardContent className="p-0">
              <div className="divide-y divide-white/5">
                {invoices?.slice(0, 5).map(inv => (
                  <div key={inv.id} className="p-4 flex items-center justify-between hover:bg-white/5 transition-colors">
                    <div>
                      <p className="font-medium text-white">{inv.customerName}</p>
                      <p className="text-sm text-muted-foreground font-mono">INV-{inv.id.toString().padStart(4, '0')}</p>
                    </div>
                    <div className="text-right">
                      <p className="font-bold text-white">${Number(inv.totalAmount).toLocaleString()}</p>
                      <p className="text-xs text-muted-foreground">{new Date(inv.date).toLocaleDateString()}</p>
                    </div>
                  </div>
                ))}
                {(!invoices || invoices.length === 0) && (
                  <div className="p-8 text-center text-muted-foreground font-mono">No financial data logged.</div>
                )}
              </div>
              <div className="p-4 border-t border-white/5 bg-black/20">
                <Link href="/admin/invoices" className="text-sm text-secondary hover:text-white transition-colors font-medium">Access Ledger →</Link>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </AdminLayout>
  );
}
