import { useState } from "react";
import { AdminLayout } from "@/components/AdminLayout";
import { useCustomers, useCreateCustomer, useDeleteCustomer } from "@/hooks/use-customers";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Trash2, Plus, Users as UsersIcon } from "lucide-react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { insertCustomerSchema, type InsertCustomer } from "@shared/routes";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";

export default function Customers() {
  const { data: customers, isLoading } = useCustomers();
  const createCustomer = useCreateCustomer();
  const deleteCustomer = useDeleteCustomer();
  const [isDialogOpen, setIsDialogOpen] = useState(false);

  const form = useForm<InsertCustomer>({
    resolver: zodResolver(insertCustomerSchema),
    defaultValues: { name: "", email: "", phone: "", company: "" }
  });

  const onSubmit = (data: InsertCustomer) => {
    createCustomer.mutate(data, {
      onSuccess: () => {
        setIsDialogOpen(false);
        form.reset();
      }
    });
  };

  return (
    <AdminLayout>
      <div className="space-y-6">
        <div className="flex justify-between items-end">
          <div>
            <h1 className="text-3xl font-display font-bold text-white uppercase tracking-wider">Customer Matrix</h1>
            <p className="text-muted-foreground">Authorized personnel and organizations.</p>
          </div>
          <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
            <DialogTrigger asChild>
              <Button className="bg-secondary hover:bg-secondary/90 text-white font-bold gap-2">
                <Plus className="w-4 h-4" /> Register Entity
              </Button>
            </DialogTrigger>
            <DialogContent className="bg-card border border-white/10 text-white">
              <DialogHeader>
                <DialogTitle className="font-display text-xl uppercase tracking-wider border-b border-white/10 pb-4">New Entity Registration</DialogTitle>
              </DialogHeader>
              <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4 pt-4">
                <div className="space-y-2">
                  <Label>Entity Name</Label>
                  <Input {...form.register("name")} className="bg-black/50 border-white/10" />
                </div>
                <div className="space-y-2">
                  <Label>Secure Email</Label>
                  <Input type="email" {...form.register("email")} className="bg-black/50 border-white/10" />
                </div>
                <div className="space-y-2">
                  <Label>Organization (Optional)</Label>
                  <Input {...form.register("company")} className="bg-black/50 border-white/10" />
                </div>
                <div className="space-y-2">
                  <Label>Comms Number (Optional)</Label>
                  <Input {...form.register("phone")} className="bg-black/50 border-white/10" />
                </div>
                <Button type="submit" className="w-full bg-secondary hover:bg-secondary/90 text-white mt-4" disabled={createCustomer.isPending}>
                  {createCustomer.isPending ? "REGISTERING..." : "COMMIT RECORD"}
                </Button>
              </form>
            </DialogContent>
          </Dialog>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {isLoading ? (
             <div className="col-span-full p-8 text-center text-muted-foreground font-mono">Accessing database...</div>
          ) : customers?.length === 0 ? (
            <div className="col-span-full p-12 text-center border border-white/10 border-dashed rounded-xl flex flex-col items-center">
              <UsersIcon className="w-12 h-12 text-muted-foreground mb-4 opacity-50" />
              <p className="text-lg text-white font-display">VOID</p>
              <p className="text-muted-foreground">No entities registered in system.</p>
            </div>
          ) : (
            customers?.map((customer) => (
              <div key={customer.id} className="glass-panel p-6 rounded-xl border border-white/10 tech-border relative group">
                <div className="absolute top-4 right-4 opacity-0 group-hover:opacity-100 transition-opacity">
                  <Button 
                    variant="ghost" 
                    size="icon" 
                    className="h-8 w-8 text-destructive hover:bg-destructive/20 hover:text-destructive"
                    onClick={() => {
                      if(confirm("Permanently erase entity?")) deleteCustomer.mutate(customer.id);
                    }}
                  >
                    <Trash2 className="w-4 h-4" />
                  </Button>
                </div>
                <div className="w-12 h-12 rounded-lg bg-white/5 border border-white/10 flex items-center justify-center mb-4">
                  <UsersIcon className="w-6 h-6 text-muted-foreground" />
                </div>
                <h3 className="text-xl font-bold text-white mb-1">{customer.name}</h3>
                {customer.company && <p className="text-sm font-mono text-secondary mb-4 uppercase">{customer.company}</p>}
                
                <div className="space-y-2 text-sm text-muted-foreground mt-4 pt-4 border-t border-white/5">
                  <div className="flex justify-between">
                    <span className="opacity-50">COMMS:</span>
                    <span className="text-white">{customer.email}</span>
                  </div>
                  {customer.phone && (
                    <div className="flex justify-between">
                      <span className="opacity-50">FREQ:</span>
                      <span className="text-white">{customer.phone}</span>
                    </div>
                  )}
                </div>
              </div>
            ))
          )}
        </div>
      </div>
    </AdminLayout>
  );
}
