import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { insertUserSchema, type InsertUser } from "@shared/routes";
import { useLogin, useUser } from "@/hooks/use-auth";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Button } from "@/components/ui/button";
import { ShieldAlert, Terminal } from "lucide-react";
import { useLocation } from "wouter";
import { useEffect } from "react";

export default function Login() {
  const [, setLocation] = useLocation();
  const { data: user } = useUser();
  const login = useLogin();

  const form = useForm<InsertUser>({
    resolver: zodResolver(insertUserSchema),
    defaultValues: { username: "", password: "" }
  });

  // If already logged in, redirect to admin
  useEffect(() => {
    if (user) {
      setLocation("/admin");
    }
  }, [user, setLocation]);

  const onSubmit = (data: InsertUser) => {
    login.mutate(data, {
      onSuccess: () => setLocation("/admin")
    });
  };

  return (
    <div className="min-h-screen bg-background flex flex-col items-center justify-center p-4 relative overflow-hidden">
      {/* Background Grid */}
      <div className="absolute inset-0 bg-[url('data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSI0MCIgaGVpZ2h0PSI0MCI+CjxyZWN0IHdpZHRoPSI0MCIgaGVpZ2h0PSI0MCIgZmlsbD0ibm9uZSIvPgo8cGF0aCBkPSJNMCAwaDQwdjQwSDB6IiBmaWxsPSJub25lIiBzdHJva2U9IiNmZmYiIHN0cm9rZS1vcGFjaXR5PSIwLjA1IiBzdHJva2Utd2lkdGg9IjEiLz4KPC9zdmc+')] z-0 opacity-50" />
      
      <div className="w-full max-w-md relative z-10">
        <div className="text-center mb-8">
          <div className="inline-flex items-center justify-center w-20 h-20 rounded-2xl bg-card border border-white/10 shadow-[0_0_30px_rgba(249,115,22,0.2)] mb-6">
            <ShieldAlert className="w-10 h-10 text-primary" />
          </div>
          <h1 className="text-3xl font-display font-bold uppercase tracking-widest text-white">System Auth</h1>
          <p className="text-muted-foreground mt-2 font-mono text-sm">RESTRICTED ACCESS ONLY</p>
        </div>

        <div className="glass-panel p-8 rounded-2xl tech-border bg-card/90 backdrop-blur-xl">
          <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
            <div className="space-y-2">
              <Label className="font-mono text-xs uppercase tracking-wider text-muted-foreground">Operator ID</Label>
              <Input 
                {...form.register("username")} 
                className="bg-black/60 border-white/10 h-12 font-mono focus-visible:ring-secondary" 
                placeholder="admin"
              />
            </div>
            
            <div className="space-y-2">
              <Label className="font-mono text-xs uppercase tracking-wider text-muted-foreground">Access Cipher</Label>
              <Input 
                type="password" 
                {...form.register("password")} 
                className="bg-black/60 border-white/10 h-12 font-mono focus-visible:ring-secondary" 
                placeholder="••••••••"
              />
            </div>

            {login.isError && (
              <div className="p-3 bg-destructive/10 border border-destructive/30 rounded text-destructive text-sm font-mono flex items-center gap-2">
                <Terminal className="w-4 h-4" /> Auth Denied
              </div>
            )}

            <Button 
              type="submit" 
              className="w-full h-12 font-display font-bold tracking-widest bg-secondary hover:bg-secondary/80 text-white mt-4"
              disabled={login.isPending}
            >
              {login.isPending ? "AUTHENTICATING..." : "INITIALIZE LOGIN"}
            </Button>
            
            <p className="text-center text-xs text-muted-foreground font-mono mt-4">
              Hint: admin / admin
            </p>
          </form>
        </div>
      </div>
    </div>
  );
}
