import { PublicLayout } from "@/components/PublicLayout";
import { useCreateServiceRequest } from "@/hooks/use-service-requests";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { insertServiceRequestSchema, type InsertServiceRequest } from "@shared/routes";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Button } from "@/components/ui/button";
import { useToast } from "@/hooks/use-toast";
import { Terminal } from "lucide-react";

export default function Contact() {
  const { toast } = useToast();
  const createRequest = useCreateServiceRequest();
  
  const form = useForm<InsertServiceRequest>({
    resolver: zodResolver(insertServiceRequestSchema),
    defaultValues: {
      name: "",
      email: "",
      phone: "",
      serviceInterest: "general",
      message: "",
    }
  });

  const onSubmit = (data: InsertServiceRequest) => {
    createRequest.mutate(data, {
      onSuccess: () => {
        toast({
          title: "Transmission Secure",
          description: "Your request has been logged. An operative will contact you shortly.",
        });
        form.reset();
      },
      onError: (error) => {
        toast({
          title: "Transmission Failed",
          description: error.message,
          variant: "destructive",
        });
      }
    });
  };

  return (
    <PublicLayout>
      <div className="container mx-auto px-4 py-24 max-w-6xl">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-16">
          
          <div className="space-y-8">
            <div className="inline-flex items-center gap-2 text-primary font-display text-sm tracking-widest uppercase">
              <Terminal className="w-4 h-4" /> Secure Channel
            </div>
            <h1 className="text-4xl md:text-5xl font-display font-bold">INITIATE <br/><span className="text-secondary">CONTACT</span></h1>
            <p className="text-lg text-muted-foreground leading-relaxed">
              Submit your operational requirements. Our engineering team will review your parameters and propose a hardened infrastructure solution.
            </p>
            
            <div className="space-y-6 pt-8 border-t border-white/10">
              <div>
                <h4 className="font-display font-bold text-white mb-1">Global Headquarters</h4>
                <p className="text-muted-foreground">Level 42, The Shard<br/>Cyber District, Nexus City</p>
              </div>
              <div>
                <h4 className="font-display font-bold text-white mb-1">Emergency Dispatch</h4>
                <p className="text-primary font-mono">+1 (800) SEC-URE9</p>
              </div>
            </div>
          </div>

          <div className="glass-panel p-8 md:p-10 rounded-2xl tech-border bg-card/80">
            <h3 className="text-2xl font-display font-bold mb-8 flex items-center gap-3 border-b border-white/10 pb-4">
              <div className="w-3 h-3 bg-primary rounded-full animate-pulse" />
              TRANSMIT DATA
            </h3>
            
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="space-y-2">
                  <Label htmlFor="name">Operative Name</Label>
                  <Input id="name" {...form.register("name")} className="bg-black/50 border-white/10 focus-visible:ring-primary" />
                  {form.formState.errors.name && <p className="text-xs text-destructive">{form.formState.errors.name.message}</p>}
                </div>
                <div className="space-y-2">
                  <Label htmlFor="email">Secure Email</Label>
                  <Input id="email" type="email" {...form.register("email")} className="bg-black/50 border-white/10 focus-visible:ring-primary" />
                  {form.formState.errors.email && <p className="text-xs text-destructive">{form.formState.errors.email.message}</p>}
                </div>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="space-y-2">
                  <Label htmlFor="phone">Contact Number (Optional)</Label>
                  <Input id="phone" {...form.register("phone")} className="bg-black/50 border-white/10 focus-visible:ring-primary" />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="serviceInterest">Primary Vector</Label>
                  <select 
                    id="serviceInterest" 
                    {...form.register("serviceInterest")}
                    className="flex h-10 w-full rounded-md border border-white/10 bg-black/50 px-3 py-2 text-sm ring-offset-background focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-primary focus-visible:ring-offset-2"
                  >
                    <option value="general">General Inquiry</option>
                    <option value="access-control">Access Control</option>
                    <option value="cctv">CCTV & Surveillance</option>
                    <option value="networking">IT Networking</option>
                  </select>
                </div>
              </div>

              <div className="space-y-2">
                <Label htmlFor="message">Mission Parameters</Label>
                <Textarea 
                  id="message" 
                  rows={5} 
                  {...form.register("message")} 
                  className="bg-black/50 border-white/10 focus-visible:ring-primary resize-none" 
                />
                {form.formState.errors.message && <p className="text-xs text-destructive">{form.formState.errors.message.message}</p>}
              </div>

              <Button 
                type="submit" 
                className="w-full h-12 font-bold tracking-wider bg-primary hover:bg-primary/90 text-primary-foreground shadow-[0_0_15px_rgba(249,115,22,0.3)]"
                disabled={createRequest.isPending}
              >
                {createRequest.isPending ? "ENCRYPTING..." : "SUBMIT DISPATCH"}
              </Button>
            </form>
          </div>

        </div>
      </div>
    </PublicLayout>
  );
}
