import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { api, buildUrl, type InsertServiceRequest } from "@shared/routes";

export function useServiceRequests() {
  return useQuery({
    queryKey: [api.serviceRequests.list.path],
    queryFn: async () => {
      const res = await fetch(api.serviceRequests.list.path, { credentials: "include" });
      if (!res.ok) throw new Error("Failed to fetch service requests");
      return api.serviceRequests.list.responses[200].parse(await res.json());
    },
  });
}

export function useCreateServiceRequest() {
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: async (data: InsertServiceRequest) => {
      const res = await fetch(api.serviceRequests.create.path, {
        method: api.serviceRequests.create.method,
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(data),
      });
      if (!res.ok) throw new Error("Failed to create request");
      return api.serviceRequests.create.responses[201].parse(await res.json());
    },
    onSuccess: () => queryClient.invalidateQueries({ queryKey: [api.serviceRequests.list.path] }),
  });
}

export function useUpdateServiceRequestStatus() {
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: async ({ id, status }: { id: number; status: string }) => {
      const url = buildUrl(api.serviceRequests.update.path, { id });
      const res = await fetch(url, {
        method: api.serviceRequests.update.method,
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ status }),
        credentials: "include",
      });
      if (!res.ok) throw new Error("Failed to update status");
      return api.serviceRequests.update.responses[200].parse(await res.json());
    },
    onSuccess: () => queryClient.invalidateQueries({ queryKey: [api.serviceRequests.list.path] }),
  });
}

export function useDeleteServiceRequest() {
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: async (id: number) => {
      const url = buildUrl(api.serviceRequests.delete.path, { id });
      const res = await fetch(url, {
        method: api.serviceRequests.delete.method,
        credentials: "include",
      });
      if (!res.ok) throw new Error("Failed to delete request");
    },
    onSuccess: () => queryClient.invalidateQueries({ queryKey: [api.serviceRequests.list.path] }),
  });
}
