import { Link, useLocation } from "wouter";
import { useUser, useLogout } from "@/hooks/use-auth";
import { useEffect } from "react";
import { 
  LayoutDashboard, 
  Users, 
  FileText, 
  Settings, 
  LogOut,
  ShieldAlert,
  Inbox
} from "lucide-react";
import { Button } from "@/components/ui/button";

export function AdminLayout({ children }: { children: React.ReactNode }) {
  const { data: user, isLoading } = useUser();
  const logout = useLogout();
  const [location] = useLocation();

  useEffect(() => {
    if (!isLoading && !user) {
      window.location.href = "/login";
    }
  }, [user, isLoading]);

  if (isLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-background">
        <div className="w-16 h-16 border-4 border-primary border-t-transparent rounded-full animate-spin"></div>
      </div>
    );
  }

  if (!user) {
    return null;
  }

  const navItems = [
    { href: "/admin", icon: LayoutDashboard, label: "Dashboard" },
    { href: "/admin/requests", icon: Inbox, label: "Service Requests" },
    { href: "/admin/customers", icon: Users, label: "Customers" },
    { href: "/admin/invoices", icon: FileText, label: "Invoices" },
    { href: "/admin/settings", icon: Settings, label: "System Settings" },
  ];

  return (
    <div className="min-h-screen flex bg-background">
      {/* Sidebar - Hidden during printing */}
      <aside className="w-64 border-r border-white/10 glass-panel flex flex-col no-print">
        <div className="h-20 flex items-center px-6 border-b border-white/10">
          <Link href="/admin" className="flex items-center gap-2 text-secondary">
            <ShieldAlert className="w-8 h-8" />
            <span className="font-display font-bold text-xl uppercase tracking-wider text-foreground">
              Command<span className="text-secondary">Center</span>
            </span>
          </Link>
        </div>

        <nav className="flex-1 py-6 px-4 space-y-2">
          {navItems.map((item) => {
            const Icon = item.icon;
            const isActive = location === item.href || (item.href !== "/admin" && location.startsWith(item.href));
            
            return (
              <Link key={item.href} href={item.href} className={`
                flex items-center gap-3 px-4 py-3 rounded-lg transition-all duration-200
                ${isActive 
                  ? "bg-secondary/20 text-secondary border border-secondary/30 shadow-[inset_0_0_10px_rgba(2,132,199,0.2)]" 
                  : "text-muted-foreground hover:text-foreground hover:bg-white/5"}
              `}>
                <Icon className="w-5 h-5" />
                <span className="font-medium">{item.label}</span>
              </Link>
            );
          })}
        </nav>

        <div className="p-4 border-t border-white/10">
          <div className="px-4 py-3 mb-4 rounded-lg bg-black/40 border border-white/5 flex flex-col">
            <span className="text-xs text-muted-foreground uppercase tracking-widest font-display">Operator</span>
            <span className="font-medium text-primary">{user.username}</span>
          </div>
          <Button 
            variant="destructive" 
            className="w-full justify-start gap-2"
            onClick={() => logout.mutate()}
            disabled={logout.isPending}
          >
            <LogOut className="w-4 h-4" />
            {logout.isPending ? "Terminating..." : "Terminate Session"}
          </Button>
        </div>
      </aside>

      {/* Main Content */}
      <main className="flex-1 flex flex-col h-screen overflow-hidden">
        <div className="flex-1 overflow-auto p-8">
          {children}
        </div>
      </main>
    </div>
  );
}
